<?php
/**
 * Plugin Name: HumanGate
 * Plugin URI: https://wordpress.org/plugins/humangate/
 * Description: Lightweight but enforceable AI & Indexing Refusal + Selective Friction layer for WordPress. Global refusal signals, emergency lockdown, and scraping cost amplification without CAPTCHAs or heavy databases.
 * Version: 1.0.0
 * Author: DarkAI
 * Author URI: https://darkai.ca/
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: humangate
 * Requires at least: 5.0
 * Requires PHP: 7.4
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('BLOCK_AI_PLUGIN_VERSION', '1.0.0');
define('BLOCK_AI_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BLOCK_AI_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BLOCK_AI_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Autoloader
spl_autoload_register(function ($class) {
    $prefix = 'BlockAI\\';
    $base_dir = BLOCK_AI_PLUGIN_DIR . 'includes/';
    
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    
    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';
    
    if (file_exists($file)) {
        require $file;
    }
});

// Initialize plugin
function block_ai_plugin_init() {
    // Load content control (must be initialized before refusal layer)
    $content_control = new BlockAI\ContentControl();
    $content_control->init();
    
    // Load core classes
    $refusal_layer = new BlockAI\RefusalLayer();
    $refusal_layer->init();
    
    $lockdown = new BlockAI\EmergencyLockdown();
    $lockdown->init();
    
    $friction = new BlockAI\SelectiveFriction();
    $friction->init();
    
    $telemetry = new BlockAI\Telemetry();
    $telemetry->init();
    
    // Admin interface
    if (is_admin()) {
        $admin = new BlockAI\Admin();
        $admin->init();
    }
}

// Initialize on plugins_loaded
add_action('plugins_loaded', 'block_ai_plugin_init');

// Activation hook
register_activation_hook(__FILE__, 'block_ai_activation');

/**
 * Activation callback - sets default options
 */
function block_ai_activation() {
    // Set default options
    add_option('block_ai_block_search_engines', true);
    add_option('block_ai_global_refusal', true);
    add_option('block_ai_enforcement_mode', 'signals_only');
    add_option('block_ai_verify_search_engine_bots', false);
    add_option('block_ai_lockdown_mode', false);
    add_option('block_ai_friction_enabled', true);
    add_option('block_ai_friction_burst_pages', 12);
    add_option('block_ai_friction_burst_seconds', 5);
    add_option('block_ai_friction_rate_pages', 40);
    add_option('block_ai_friction_rate_seconds', 60);
    add_option('block_ai_lockdown_require_login', false);
    add_option('block_ai_disable_robots_txt', false);
    add_option('block_ai_stats_retention', 'weekly');
    
    // Set default post type settings (all enabled by default)
    // Use function_exists check and validate result to avoid fatal errors
    if (function_exists('get_post_types')) {
        $post_types = get_post_types(['public' => true], 'names');
        if (!empty($post_types) && is_array($post_types)) {
            foreach ($post_types as $post_type) {
                if (!empty($post_type)) {
                    add_option('block_ai_refusal_post_types_' . $post_type, true);
                }
            }
        }
    }
}

// Deactivation hook
register_deactivation_hook(__FILE__, function() {
    // Clean up transients and scheduled events
    delete_transient('block_ai_daily_counters');
    delete_transient('block_ai_weekly_counters');
    wp_clear_scheduled_hook('block_ai_weekly_cleanup');
});

// Uninstall hook - removes all plugin data when plugin is deleted
register_uninstall_hook(__FILE__, 'block_ai_uninstall');

/**
 * Clean up all plugin data on uninstall
 * 
 * This function removes all options, transients, scheduled events, and post meta
 * created by HumanGate when the plugin is deleted.
 */
function block_ai_uninstall() {
    // Security check: ensure this is being called from WordPress
    if (!defined('WP_UNINSTALL_PLUGIN')) {
        return;
    }
    
    // Delete all plugin options
    $options = [
        'block_ai_block_search_engines',
        'block_ai_global_refusal',
        'block_ai_enforcement_mode',
        'block_ai_verify_search_engine_bots',
        'block_ai_lockdown_mode',
        'block_ai_friction_enabled',
        'block_ai_friction_burst_pages',
        'block_ai_friction_burst_seconds',
        'block_ai_friction_rate_pages',
        'block_ai_friction_rate_seconds',
        'block_ai_lockdown_require_login',
        'block_ai_disable_robots_txt',
        'block_ai_stats_retention',
    ];
    
    foreach ($options as $option) {
        delete_option($option);
    }
    
    // Delete post type settings (only if WordPress is loaded)
    if (function_exists('get_post_types')) {
        $post_types = get_post_types(['public' => true], 'names');
        if (is_array($post_types)) {
            foreach ($post_types as $post_type) {
                if ($post_type) {
                    delete_option('block_ai_refusal_post_types_' . $post_type);
                }
            }
        }
    }
    
    // Clear scheduled cron events
    if (function_exists('wp_clear_scheduled_hook')) {
        wp_clear_scheduled_hook('block_ai_weekly_cleanup');
    }
    
    // Note: Transients will expire automatically and don't need manual deletion
    // Post meta can be left as it doesn't cause issues if orphaned
}

